{*******************************************************}
{               RichEdit Syntax HighLight               }
{                     version 3.0                       }
{ Author:                                               }
{ Serhiy Perevoznyk                                     }
{ serge_perevoznyk@hotmail.com                          }
{                                                       }
{*******************************************************}

{The contents of this file are subject to the Mozilla Public License
Version 1.1 (the "License"); you may not use this file except in compliance
with the License. You may obtain a copy of the License at
http://www.mozilla.org/MPL/

Software distributed under the License is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
the specific language governing rights and limitations under the License.

The Original Code is: SynHighlighterPHP.pas, released 2000-04-21.
The Original Code is based on the wmPHPSyn.pas file from the
mwEdit component suite by Martin Waldenburg and other developers, the Initial
Author of this file is Willo van der Merwe.
All Rights Reserved.
The Original Code can be obtained from http://synedit.sourceforge.net/
}

unit psvPHPScript;

interface

uses
  SysUtils, 
  Windows, 
  Messages, 
  Classes, 
  Controls, 
  Graphics, 
  psvRichSyntax;

type
  TtkTokenKind = (tkComment, tkIdentifier, tkInvalidSymbol, tkKey, tkNull,
    tkNumber, tkSpace, tkString, tkSymbol, tkUnknown, tkVariable);

  TRangeState = (rsUnKnown, rsString39, rsString34, rsComment);

  TProcTableProc = procedure of object;

  PIdentFuncTableFunc = ^TIdentFuncTableFunc;
  TIdentFuncTableFunc = function: TtkTokenKind of object;

type
  TpsvPHPRTF = class(TpsvRTFSyntax)
  private
    fRange: TRangeState;
    fLine: PChar;
    fLineNumber: Integer;
    fProcTable: array[#0..#255] of TProcTableProc;
    Run: LongInt;
    fStringLen: Integer;
    fToIdent: PChar;
    fTokenPos: Integer;
    FTokenID: TtkTokenKind;
    fIdentFuncTable: array[0..206] of TIdentFuncTableFunc;
    function KeyHash(ToHash: PChar): Integer;
    function KeyComp(const aKey: String): Boolean;
    function Func15: TtkTokenKind;
    function Func18: TtkTokenKind;
    function Func19: TtkTokenKind;
    function Func28: TtkTokenKind;
    function Func31: TtkTokenKind;
    function Func33: TtkTokenKind;
    function Func36: TtkTokenKind;
    function Func37: TtkTokenKind;
    function Func38: TtkTokenKind;
    function Func39: TtkTokenKind;
    function Func40: TtkTokenKind;
    function Func41: TtkTokenKind;
    function Func42: TtkTokenKind;
    function Func43: TtkTokenKind;
    function Func49: TtkTokenKind;
    function Func54: TtkTokenKind;
    function Func55: TtkTokenKind;
    function Func56: TtkTokenKind;
    function Func57: TtkTokenKind;
    function Func58: TtkTokenKind;
    function Func59: TtkTokenKind;
    function Func60: TtkTokenKind;
    function Func62: TtkTokenKind;
    function Func63: TtkTokenKind;
    function Func64: TtkTokenKind;
    function Func68: TtkTokenKind;
    function Func69: TtkTokenKind;
    function Func71: TtkTokenKind;
    function Func72: TtkTokenKind;
    function Func77: TtkTokenKind;
    function Func78: TtkTokenKind;
    function Func79: TtkTokenKind;
    function Func80: TtkTokenKind;
    function Func82: TtkTokenKind;
    function Func87: TtkTokenKind;
    function Func91: TtkTokenKind;
    function Func93: TtkTokenKind;
    function Func96: TtkTokenKind;
    function Func101: TtkTokenKind;
    function Func102: TtkTokenKind;
    function Func105: TtkTokenKind;
    function Func151: TtkTokenKind;
    function Func156: TtkTokenKind;
    function Func164: TtkTokenKind;
    function Func177: TtkTokenKind;
    function Func206: TtkTokenKind;
    procedure AndSymbolProc;
    procedure AsciiCharProc;
    procedure AtSymbolProc;
    procedure BraceCloseProc;
    procedure BraceOpenProc;
    procedure CRProc;
    procedure ColonProc;
    procedure CommaProc;
    procedure EqualProc;
    procedure GreaterProc;
    procedure IdentProc;
    procedure LFProc;
    procedure LowerProc;
    procedure MinusProc;
    procedure MultiplyProc;
    procedure NotSymbolProc;
    procedure NullProc;
    procedure NumberProc;
    procedure OrSymbolProc;
    procedure PlusProc;
    procedure PointProc;
    procedure PoundProc;
    procedure QuestionProc;
    procedure RemainderSymbolProc;
    procedure RoundCloseProc;
    procedure RoundOpenProc;
    procedure SemiColonProc;
    procedure SlashProc;
    procedure SpaceProc;
    procedure SquareCloseProc;
    procedure SquareOpenProc;
    procedure StringProc;
    procedure TildeProc;
    procedure VariableProc;
    procedure XOrSymbolProc;
    procedure UnknownProc;
    function AltFunc: TtkTokenKind;
    procedure InitIdent;
    function IdentKind(MayBe: PChar): TtkTokenKind;
    procedure MakeMethodTables;
    procedure AnsiCProc;
    procedure String39Proc;
    procedure String34Proc;
  protected
    function GetEOL: Boolean; override;
    function GetRange: Pointer; 
    function GetTokenID: TtkTokenKind;
    procedure SetLine(NewValue: String; LineNumber: Integer); override;
    function GetToken: String; override;
    function GetTokenAttribute: integer; override;
    function GetTokenKind: integer; 
    function GetTokenPos: Integer; 
    procedure Next; override;
    procedure SetRange(Value: Pointer); 
    procedure ResetRange; 
  public
    constructor Create; override;
    procedure SetupDefaultColors; override;
  end;

implementation


var
  Identifiers: array[#0..#255] of ByteBool;
  mHashTable: array[#0..#255] of Integer;

procedure MakeIdentTable;
var
  I, J: Char;
begin
  for I := #0 to #255 do
  begin
    Case I of
      '_', '0'..'9', 'a'..'z', 'A'..'Z': Identifiers[I] := True;
    else Identifiers[I] := False;
    end;
    J := UpCase(I);
    Case I in ['_', 'A'..'Z', 'a'..'z'] of
      True: mHashTable[I] := Ord(J) - 64
    else mHashTable[I] := 0;
    end;
  end;
end;

procedure TpsvPHPRTF.InitIdent;
var
  I: Integer;
  pF: PIdentFuncTableFunc;
begin
  pF := PIdentFuncTableFunc(@fIdentFuncTable);
  for I := Low(fIdentFuncTable) to High(fIdentFuncTable) do begin
    pF^ := AltFunc;
    Inc(pF);
  end;
  fIdentFuncTable[15] := Func15;
  fIdentFuncTable[18] := Func18;
  fIdentFuncTable[19] := Func19;
  fIdentFuncTable[28] := Func28;
  fIdentFuncTable[31] := Func31;
  fIdentFuncTable[33] := Func33;
  fIdentFuncTable[36] := Func36;
  fIdentFuncTable[37] := Func37;
  fIdentFuncTable[38] := Func38;
  fIdentFuncTable[39] := Func39;
  fIdentFuncTable[40] := Func40;
  fIdentFuncTable[41] := Func41;
  fIdentFuncTable[42] := Func42;
  fIdentFuncTable[43] := Func43;
  fIdentFuncTable[49] := Func49;
  fIdentFuncTable[54] := Func54;
  fIdentFuncTable[55] := Func55;
  fIdentFuncTable[56] := Func56;
  fIdentFuncTable[57] := Func57;
  fIdentFuncTable[58] := Func58;
  fIdentFuncTable[59] := Func59;
  fIdentFuncTable[60] := Func60;
  fIdentFuncTable[62] := Func62;
  fIdentFuncTable[63] := Func63;
  fIdentFuncTable[64] := Func64;
  fIdentFuncTable[68] := Func68;
  fIdentFuncTable[69] := Func69;
  fIdentFuncTable[71] := Func71;
  fIdentFuncTable[72] := Func72;
  fIdentFuncTable[77] := Func77;
  fIdentFuncTable[78] := Func78;
  fIdentFuncTable[79] := Func79;
  fIdentFuncTable[80] := Func80;
  fIdentFuncTable[82] := Func82;
  fIdentFuncTable[87] := Func87;
  fIdentFuncTable[91] := Func91;
  fIdentFuncTable[93] := Func93;
  fIdentFuncTable[96] := Func96;
  fIdentFuncTable[101] := Func101;
  fIdentFuncTable[102] := Func102;
  fIdentFuncTable[105] := Func105;
  fIdentFuncTable[151] := Func151;
  fIdentFuncTable[156] := Func156;
  fIdentFuncTable[164] := Func164;
  fIdentFuncTable[177] := Func177;
  fIdentFuncTable[206] := Func206;
end;

function TpsvPHPRTF.KeyHash(ToHash: PChar): Integer;
begin
  Result := 0;
  while ToHash^ in ['_', '0'..'9', 'a'..'z', 'A'..'Z'] do
  begin
    inc(Result, mHashTable[ToHash^]);
    inc(ToHash);
  end;
  fStringLen := ToHash - fToIdent;
end;

function TpsvPHPRTF.KeyComp(const aKey: String): Boolean;
var
  I: Integer;
  Temp: PChar;
begin
  Temp := fToIdent;
  if Length(aKey) = fStringLen then
  begin
    Result := True;
    for i := 1 to fStringLen do
    begin
      if mHashTable[Temp^] <> mHashTable[aKey[i]] then
      begin
        Result := False;
        break;
      end;
      inc(Temp);
    end;
  end else Result := False;
end;

function TpsvPHPRTF.Func15: TtkTokenKind;
begin
  if KeyComp('if') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func18: TtkTokenKind;
begin
  if KeyComp('die') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func19: TtkTokenKind;
begin
  if KeyComp('and') then Result := tkKey else
    if KeyComp('do') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func28: TtkTokenKind;
begin
  if KeyComp('case') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func31: TtkTokenKind;
begin
  if KeyComp('echo') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func33: TtkTokenKind;
begin
  if KeyComp('or') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func36: TtkTokenKind;
begin
  if KeyComp('real') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func37: TtkTokenKind;
begin
  if KeyComp('break') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func38: TtkTokenKind;
begin
  if KeyComp('endif') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func39: TtkTokenKind;
begin
  if KeyComp('for') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func40: TtkTokenKind;
begin
  if KeyComp('eval') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func41: TtkTokenKind;
begin
  if KeyComp('var') then Result := tkKey else
    if KeyComp('else') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func42: TtkTokenKind;
begin
  if KeyComp('new') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func43: TtkTokenKind;
begin
  if KeyComp('false') then Result := tkKey else
    if KeyComp('int') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func49: TtkTokenKind;
begin
  if KeyComp('global') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func54: TtkTokenKind;
begin
  if KeyComp('float') then Result := tkKey else
    if KeyComp('class') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func55: TtkTokenKind;
begin
  if KeyComp('object') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func56: TtkTokenKind;
begin
  if KeyComp('elseif') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func57: TtkTokenKind;
begin
  if KeyComp('while') then Result := tkKey else
    if KeyComp('xor') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func58: TtkTokenKind;
begin
  if KeyComp('exit') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func59: TtkTokenKind;
begin
  if KeyComp('double') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func60: TtkTokenKind;
begin
  if KeyComp('list') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func62: TtkTokenKind;
begin
  if KeyComp('endfor') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func63: TtkTokenKind;
begin
  if KeyComp('array') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func64: TtkTokenKind;
begin
  if KeyComp('true') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func68: TtkTokenKind;
begin
  if KeyComp('include') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func69: TtkTokenKind;
begin
  if KeyComp('default') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func71: TtkTokenKind;
begin
  if KeyComp('const') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func72: TtkTokenKind;
begin
  if KeyComp('isset') then Result := tkKey else
    if KeyComp('static') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func77: TtkTokenKind;
begin
  if KeyComp('print') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func78: TtkTokenKind;
begin
  if KeyComp('integer') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func79: TtkTokenKind;
begin
  if KeyComp('empty') then Result := tkKey else
    if KeyComp('unset') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func80: TtkTokenKind;
begin
  if KeyComp('endwhile') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func82: TtkTokenKind;
begin
  if KeyComp('switch') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func87: TtkTokenKind;
begin
  if KeyComp('string') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func91: TtkTokenKind;
begin
  if KeyComp('extends') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func93: TtkTokenKind;
begin
  if KeyComp('require') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func96: TtkTokenKind;
begin
  if KeyComp('return') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func101: TtkTokenKind;
begin
  if KeyComp('continue') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func102: TtkTokenKind;
begin
  if KeyComp('function') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func105: TtkTokenKind;
begin
  if KeyComp('cfunction') then Result := tkKey else
    if KeyComp('endswitch') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func151: TtkTokenKind;
begin
  if KeyComp('highlight_file') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func156: TtkTokenKind;
begin
  if KeyComp('__FILE__') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func164: TtkTokenKind;
begin
  if KeyComp('old_function') then Result := tkKey else
    if KeyComp('__LINE__') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func177: TtkTokenKind;
begin
  if KeyComp('show_source') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.Func206: TtkTokenKind;
begin
  if KeyComp('highlight_string') then Result := tkKey else Result := tkIdentifier;
end;

function TpsvPHPRTF.AltFunc: TtkTokenKind;
begin
  Result := tkIdentifier;
end;

function TpsvPHPRTF.IdentKind(MayBe: PChar): TtkTokenKind;
var
  HashKey: Integer;
begin
  fToIdent := MayBe;
  HashKey := KeyHash(MayBe);
  if HashKey < 207 then Result := fIdentFuncTable[HashKey] else Result := tkIdentifier;
end;

procedure TpsvPHPRTF.MakeMethodTables;
var
  I: Char;
begin
  for I := #0 to #255 do
    case I of
      '&': fProcTable[I] := AndSymbolProc;
      #39: fProcTable[I] := AsciiCharProc;
      '@': fProcTable[I] := AtSymbolProc;
      '}': fProcTable[I] := BraceCloseProc;
      '{': fProcTable[I] := BraceOpenProc;
      #13: fProcTable[I] := CRProc;
      ':': fProcTable[I] := ColonProc;
      ',': fProcTable[I] := CommaProc;
      '=': fProcTable[I] := EqualProc;
      '>': fProcTable[I] := GreaterProc;
      'A'..'Z', 'a'..'z', '_': fProcTable[I] := IdentProc;
      #10: fProcTable[I] := LFProc;
      '<': fProcTable[I] := LowerProc;
      '-': fProcTable[I] := MinusProc;
      '*': fProcTable[I] := MultiplyProc;
      '!': fProcTable[I] := NotSymbolProc;
      #0: fProcTable[I] := NullProc;
      '0'..'9': fProcTable[I] := NumberProc;
      '|': fProcTable[I] := OrSymbolProc;
      '+': fProcTable[I] := PlusProc;
      '.': fProcTable[I] := PointProc;
      '#': fProcTable[I] := PoundProc;
      '?': fProcTable[I] := QuestionProc;
      '%': fProcTable[I] := RemainderSymbolProc;
      ')': fProcTable[I] := RoundCloseProc;
      '(': fProcTable[I] := RoundOpenProc;
      ';': fProcTable[I] := SemiColonProc;
      '/': fProcTable[I] := SlashProc;
      #1..#9, #11, #12, #14..#32: fProcTable[I] := SpaceProc;
      ']': fProcTable[I] := SquareCloseProc;
      '[': fProcTable[I] := SquareOpenProc;
      #34: fProcTable[I] := StringProc;
      '~': fProcTable[I] := TildeProc;
      '$': fProcTable[I] := VariableProc;
      '^': fProcTable[I] := XOrSymbolProc;
      else fProcTable[I] := UnknownProc;
    end;
end;

constructor TpsvPHPRTF.Create;
begin
  inherited Create;
  InitIdent;
  MakeMethodTables;
  fRange := rsUnknown;
  CreateColorTable([clBlue,     //1   tkComment
                    clBlack,    //2   tkIdentifier
                    clRed,      //3   tkInvalidSymbol
                    clNavy,     //4   tkKey
                    clGreen,    //5   tkNumber
                    clBlack,    //6   tkSpace
                    clMaroon,   //7   tkString
                    clBlack,    //8   tkSymbol
                    clBlack,    //9   tkVariable
                    clBlack,    //10  tkUnknown
                    clBlack]);  //11

end;

procedure TpsvPHPRTF.SetLine(NewValue: String; LineNumber: Integer);
begin
  fLine := PChar(NewValue);
  Run := 0;
  fLineNumber := LineNumber;
  Next;
end;

procedure TpsvPHPRTF.AndSymbolProc;
begin
  case FLine[Run + 1] of
    '=':                               {and assign}
      begin
        inc(Run, 2);
        fTokenID := tkSymbol;
      end;
    '&':                               {conditional and}
      begin
        inc(Run, 2);
        fTokenID := tkSymbol;
      end;
  else                                 {and}
    begin
      inc(Run);
      fTokenID := tkSymbol;
    end;
  end;
end;

procedure TpsvPHPRTF.AsciiCharProc;
begin
  fRange := rsString39;
  fTokenID := tkString;
  repeat
    case FLine[Run] of
      #0, #10, #13: break;
    end;
    inc(Run);
  until FLine[Run] = #39;
  if (FLine[Run] = #39) then
    fRange := rsUnKnown;
  if FLine[Run] <> #0 then inc(Run);
end;

procedure TpsvPHPRTF.AtSymbolProc;
begin
  fTokenID := tkInvalidSymbol;
  inc(Run);
end;

procedure TpsvPHPRTF.BraceCloseProc;
begin
  inc(Run);
  fTokenId := tkSymbol;
end;

procedure TpsvPHPRTF.BraceOpenProc;
begin
  inc(Run);
  fTokenId := tkSymbol;
end;

procedure TpsvPHPRTF.CRProc;
begin
  fTokenID := tkSpace;
  Case FLine[Run + 1] of
    #10: inc(Run, 2);
  else inc(Run);
  end;
end;

procedure TpsvPHPRTF.ColonProc;
begin
  inc(Run);                            {colon - conditional}
  fTokenID := tkSymbol;
end;

procedure TpsvPHPRTF.CommaProc;
begin
  inc(Run);
  fTokenID := tkSymbol;
end;

procedure TpsvPHPRTF.EqualProc;
begin
  case FLine[Run + 1] of
    '=':                               {logical equal}
      begin
        inc(Run, 2);
        fTokenID := tkSymbol;
      end;
    '>':                               {Hash operator}
      begin
        inc(Run, 2);
        fTokenID := tkSymbol;
      end;
  else                                 {assign}
    begin
      inc(Run);
      fTokenID := tkSymbol;
    end;
  end;
end;

procedure TpsvPHPRTF.GreaterProc;
begin
  Case FLine[Run + 1] of
    '=':                               {greater than or equal to}
      begin
        inc(Run, 2);
        fTokenID := tkSymbol;
      end;
    '>':
      begin
        inc(Run, 2);
        fTokenID := tkSymbol;
      end;
  else                                 {greater than}
    begin
      inc(Run);
      fTokenID := tkSymbol;
    end;
  end;
end;

procedure TpsvPHPRTF.IdentProc;
begin
  fTokenID := IdentKind((fLine + Run));
  inc(Run, fStringLen);
  while Identifiers[fLine[Run]] do inc(Run);
end;

procedure TpsvPHPRTF.LFProc;
begin
  fTokenID := tkSpace;
  inc(Run);
end;

procedure TpsvPHPRTF.LowerProc;
begin
  case FLine[Run + 1] of
    '=':                               {less than or equal to}
      begin
        inc(Run, 2);
        fTokenID := tkSymbol;
      end;
    '<':
      begin
        if FLine[Run + 2] = '=' then   {shift left assign}
        begin
          inc(Run, 3)
        end
        else                           {shift left}
        begin
          inc(Run, 2);
        end;
        fTokenID := tkSymbol;
      end;
  else                                 {less than}
    begin
      inc(Run);
      fTokenID := tkSymbol;
    end;
  end;
end;

procedure TpsvPHPRTF.MinusProc;
begin
  case FLine[Run + 1] of
    '=':                               {subtract assign}
      begin
        inc(Run, 2);
        fTokenID := tkSymbol;
      end;
    '-':                               {decrement}
      begin
        inc(Run, 2);
        fTokenID := tkSymbol;
      end;
    '>':                               {Class operator}
      begin
        inc(Run, 2);
        fTokenID := tkSymbol;
      end;
  else                                 {subtract}
    begin
      inc(Run);
      fTokenID := tkSymbol;
    end;
  end;
end;

procedure TpsvPHPRTF.MultiplyProc;
begin
  case FLine[Run + 1] of
    '=':                               {multiply assign}
      begin
        inc(Run, 2);
        fTokenID := tkSymbol;
      end;
  else                                 {multiply}
    begin
      inc(Run);
      fTokenID := tkSymbol;
    end;
  end;
end;

procedure TpsvPHPRTF.NotSymbolProc;
begin
  case FLine[Run + 1] of
    '=':                               {not equal}
      begin
        inc(Run, 2);
        fTokenID := tkSymbol;
      end;
  else                                 {logical complement}
    begin
      inc(Run);
      fTokenID := tkSymbol;
    end;
  end;
end;

procedure TpsvPHPRTF.NullProc;
begin
  fTokenID := tkNull;
end;

procedure TpsvPHPRTF.NumberProc;
begin
  inc(Run);
  fTokenID := tkNumber;
  while FLine[Run] in
      ['0'..'9', '.', '-', 'l', 'L', 'x', 'X', 'A'..'F', 'a'..'f'] do
  begin
    case FLine[Run] of
      '.':
        if FLine[Run + 1] = '.' then break;
    end;
    inc(Run);
  end;
end;

procedure TpsvPHPRTF.OrSymbolProc;
begin
  case FLine[Run + 1] of
    '=':                               {inclusive or assign}
      begin
        inc(Run, 2);
        fTokenID := tkSymbol;
      end;
    '|':                               {conditional or}
      begin
        inc(Run, 2);
        fTokenID := tkSymbol;
      end;
  else                                 {inclusive or}
    begin
      inc(Run);
      fTokenID := tkSymbol;
    end;
  end;
end;

procedure TpsvPHPRTF.PlusProc;
begin
  case FLine[Run + 1] of
    '=':                               {add assign}
      begin
        inc(Run, 2);
        fTokenID := tkSymbol;
      end;
    '+':                               {increment}
      begin
        inc(Run, 2);
        fTokenID := tkSymbol;
      end;
  else                                 {add}
    begin
      inc(Run);
      fTokenID := tkSymbol;
    end;
  end;
end;

procedure TpsvPHPRTF.PointProc;
begin
  inc(Run);                            {point}
  fTokenID := tkSymbol;
end;

procedure TpsvPHPRTF.PoundProc;
begin
  repeat
    inc(Run);
  until FLine[Run] in [#0, #10, #13];
  fTokenID := tkComment;
end;

procedure TpsvPHPRTF.QuestionProc;
begin
  fTokenID := tkSymbol;                {question mark - conditional}
  inc(Run);
end;

procedure TpsvPHPRTF.RemainderSymbolProc;
begin
  case FLine[Run + 1] of
    '=':                               {remainder assign}
      begin
        inc(Run, 2);
        fTokenID := tkSymbol;
      end;
  else                                 {remainder}
    begin
      inc(Run);
      fTokenID := tkSymbol;
    end;
  end;
end;

procedure TpsvPHPRTF.RoundCloseProc;
begin
  inc(Run);
  fTokenID := tkSymbol;
end;

procedure TpsvPHPRTF.RoundOpenProc;
begin
  inc(Run);
  FTokenID := tkSymbol;
end;

procedure TpsvPHPRTF.SemiColonProc;
begin
  inc(Run);                            {semicolon}
  fTokenID := tkSymbol;
end;

procedure TpsvPHPRTF.SlashProc;
begin
  case FLine[Run + 1] of
    '/':                               {c++ style comments}
      begin
        inc(Run, 2);
        fTokenID := tkComment;
        while FLine[Run] <> #0 do
        begin
          case FLine[Run] of
            #10, #13: break;
          end;
          inc(Run);
        end;
      end;
    '*':
      begin
        fRange := rsComment;
        inc(Run);
        fTokenID := tkComment;       {c style comment}

        inc(Run);
        while fLine[Run] <> #0 do
          case fLine[Run] of
            '*':
              if fLine[Run + 1] = '/' then
              begin
                fRange := rsUnKnown;
                inc(Run, 2);
                break;
              end else inc(Run);
            #10: break;
            #13: break;
          else inc(Run);
          end;
      end;
    '=':                               {division assign}
      begin
        inc(Run, 2);
        fTokenID := tkSymbol;
      end;
  else                                 {division}
    begin
      inc(Run);
      fTokenID := tkSymbol;
    end;
  end;
end;

procedure TpsvPHPRTF.SpaceProc;
begin
  inc(Run);
  fTokenID := tkSpace;
  while FLine[Run] in [#1..#9, #11, #12, #14..#32] do inc(Run);
end;

procedure TpsvPHPRTF.SquareCloseProc;
begin
  inc(Run);
  fTokenID := tkSymbol;
end;

procedure TpsvPHPRTF.SquareOpenProc;
begin
  inc(Run);
  fTokenID := tkSymbol;
end;

procedure TpsvPHPRTF.StringProc;
begin
  fRange := rsString34;
  fTokenID := tkString;
  repeat
    case FLine[Run] of
      #0, #10, #13: break;
      #92:                             {backslash}
        {if we have an escaped quote it doesn't count}
        if FLine[Run + 1] = #34 then inc(Run);
    end;
    inc(Run);
  until FLine[Run] = #34;
  if (FLine[Run] = #34) then
    fRange := rsUnKnown;
  if FLine[Run] <> #0 then inc(Run);
end;

procedure TpsvPHPRTF.TildeProc;
begin
  inc(Run);                            {bitwise complement}
  fTokenId := tkSymbol;
end;

procedure TpsvPHPRTF.VariableProc;
begin
  fTokenID := tkVariable;
  inc(Run);                            
  while Identifiers[fLine[Run]] do inc(Run);
end;

procedure TpsvPHPRTF.XOrSymbolProc;
begin
  Case FLine[Run + 1] of
    '=':                               {xor assign}
      begin
        inc(Run, 2);
        fTokenID := tkSymbol;
      end;
  else                                 {xor}
    begin
      inc(Run);
      fTokenID := tkSymbol;
    end;
  end;
end;

procedure TpsvPHPRTF.UnknownProc;
begin
  inc(Run);
  fTokenID := tkUnknown;
end;

procedure TpsvPHPRTF.AnsiCProc;
begin
  fTokenID := tkComment;
  case FLine[Run] of
    #0:
      begin
        NullProc;
        exit;
      end;
    #10:
      begin
        LFProc;
        exit;
      end;
    #13:
      begin
        CRProc;
        exit;
      end;
  end;

  while FLine[Run] <> #0 do
    case FLine[Run] of
      '*':
        if fLine[Run + 1] = '/' then
        begin
          inc(Run, 2);
          fRange := rsUnKnown;
          break;
        end
        else inc(Run);
      #10: break;
      #13: break;
    else inc(Run);
    end;
end;

procedure TpsvPHPRTF.String39Proc;
begin
  fTokenID := tkString;
  case FLine[Run] of
    #0:
      begin
        NullProc;
        exit;
      end;
    #10:
      begin
        LFProc;
        exit;
      end;
    #13:
      begin
        CRProc;
        exit;
      end;
  end;

  while FLine[Run] <> #0 do
    case FLine[Run] of
      #39:
        begin
          inc(Run);
          fRange := rsUnKnown;
          break;
        end;
      #10: break;
      #13: break;
    else inc(Run);
    end;
end;

procedure TpsvPHPRTF.String34Proc;
begin
  fTokenID := tkString;
  case FLine[Run] of
    #0:
      begin
        NullProc;
        exit;
      end;
    #10:
      begin
        LFProc;
        exit;
      end;
    #13:
      begin
        CRProc;
        exit;
      end;
  end;

  while FLine[Run] <> #0 do
    case FLine[Run] of
      #34:
        begin
          inc(Run);
          fRange := rsUnKnown;
          break;
        end;
      #10: break;
      #13: break;
      #92:                             {backslash}
        {if we have an escaped quote it doesn't count}
        if FLine[Run + 1] = #34 then inc(Run, 2)
        else inc(Run);
    else inc(Run);
    end;
end;

procedure TpsvPHPRTF.Next;
begin
  fTokenPos := Run;
  case fRange of
    rsComment: AnsiCProc;
    rsString39: String39Proc;
    rsString34: String34Proc;
    else begin
      fRange := rsUnknown;
      fProcTable[fLine[Run]];
    end;
  end;
end;


function TpsvPHPRTF.GetEOL: Boolean;
begin
  Result := fTokenID = tkNull;
end;

function TpsvPHPRTF.GetRange: Pointer;
begin
  Result := Pointer(fRange);
end;

function TpsvPHPRTF.GetToken: String;
var
  Len: LongInt;
begin
  Len := Run - fTokenPos;
  SetString(Result, (FLine + fTokenPos), Len);
end;

function TpsvPHPRTF.GetTokenID: TtkTokenKind;
begin
  Result := fTokenId;
end;

function TpsvPHPRTF.GetTokenAttribute: integer;
begin
  case GetTokenID of
    tkComment: Result := 1;
    tkIdentifier: Result := 2;
    tkInvalidSymbol: Result := 3;
    tkKey: Result := 4;
    tkNumber: Result := 5;
    tkSpace: Result := 6;
    tkString: Result := 7;
    tkSymbol: Result := 8;
    tkVariable: Result := 9;
    tkUnknown: Result := 10;
    else Result := 11;
  end;
end;

function TpsvPHPRTF.GetTokenKind: integer;
begin
  Result := Ord(fTokenId);
end;

function TpsvPHPRTF.GetTokenPos: Integer;
begin
  Result := fTokenPos;
end;

procedure TpsvPHPRTF.ResetRange;
begin
  fRange := rsUnknown;
end;

procedure TpsvPHPRTF.SetRange(Value: Pointer);
begin
  fRange := TRangeState(Value);
end;



procedure TpsvPHPRTF.SetupDefaultColors;
begin
  CreateColorTable([clBlue,     //1   tkComment
                    clBlack,    //2   tkIdentifier
                    clRed,      //3   tkInvalidSymbol
                    clNavy,     //4   tkKey
                    clGreen,    //5   tkNumber
                    clBlack,    //6   tkSpace
                    clMaroon,   //7   tkString
                    clBlack,    //8   tkSymbol
                    clBlack,    //9   tkVariable
                    clBlack,    //10  tkUnknown
                    clBlack]);  //11
end;

initialization
  MakeIdentTable;
end.

